/*
 * Copyright (C) 2014 Advanced Driver Information Technology GmbH
 * Written by Norbert Uetrecht <nuetrecht@de.adit-jv.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _CAAM_EE_H
#define _CAAM_EE_H

#include <linux/kernel.h>
#include <linux/caam_key.h>

#define CAAM_EE_DEV_NAME "caamee"

/*
 * Return the kernel version information
 * this IOCTL must not be changed as it is used to check compatibility of
 * kernel and user-space
 *
 * Corresponding data structure : struct caam_ee_version
 *
 * Error codes:
 * -EFAULT      : Accessing ioctl data structure failed
 */
#define CAAM_GET_KERNEL_VERSION_OLD \
	_IOR('l', 0, struct caam_ee_version *)
#define CAAM_GET_KERNEL_VERSION \
	_IOR('l', 0, struct caam_ee_version)

/*
 * wrap unwrap ioctls
 * Wrap or unwrap a (part of a) message
 * All but the last ioctl of each msg (CAAM_OP_SINGLE, CAAM_OP_MULTI_FINALIZE)
 * must provide data lengths aligned to the block_size (i.e 16byte).
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is not set the
 * encrypted/decrypted  output of the request is returned in the same request
 * (i.e. driver waits for CAAM to finish processing)
 * For each request the out_len has to be sufficient to store the output data
 * of this request.
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is set this is not true.
 * The ioctl of a multi request processing is left after scheduling the request.
 * This allows user-space to provide the second request while the first is still
 * processed. As soon as second request is scheduled the driver waits for the
 * first request to complete and returns the output data. Afterwards the
 * user-space can proceed with the third request. Only in the last request
 * (i.e. CAAM_OP_MULTI_FINALIZE) the ioctl waits for both requests to finish.
 * Therefore the out_len has to be sufficient to hold the data of the last and
 * the second to last request. For all other requests the out_len has to be
 * sufficient to store data of the preceding request. For the first request
 * the out_len might even be 0.
 * On successful completion of the ioctl the returned out_len is the actual
 * amount of bytes written to the out buffer.
 * The easiest way to handle interleaved requests is to use a buffer sufficient
 * to store the complete output data in user-space and provide the next-data
 * pointer via out and the remaining length in bytes in out_len.
 * On each successful request the out pointer needs to be incremented by the
 * returned out_len, while the remaining length is decremented accordingly.
 * Unless CAAM_EE_FLAG_NO_CLEAR_BIT is specified the confidential temporary
 * data (i.e. input data for wrap, output data for unwrap) is overwritten in
 * kernel memory after each ioctl. In case the bit is specified corresponding
 * data buffers are only overwritten after the final call or in case of abort.
 *
 * Corresponding data structure : struct caam_ee_wrap_unwrap_params
 *
 * Flags :
 *   Operation flags:
 *     CAAM_OP_SINGLE
 *     CAAM_OP_MULTI_INIT
 *     CAAM_OP_MULTI_UPDATE
 *     CAAM_OP_MULTI_FINALIZE
 *   Algorithm:
 *     CAAM_EE_CIPHER_AES_FLAG
 *   Block-mode:
 *     CAAM_EE_WRAP_BLOCK_CCM_FLAG
 *   InData specification :
 *     CAAM_EE_WRAP_INDATA_NORM_FLAG - normal data
 *     CAAM_EE_WRAP_INDATA_AAD_FLAG - additional authenticated data
 *   Additional :
 *     CAAM_EE_FLAG_NO_CLEAR_BIT - skip overwriting confidential data
 *     CAAM_EE_FLAG_INTERLEAVED_OP_BIT - activate interleaved operation
 *
 * Error codes:
 * -ENOKEY      : No key set for current session
 * -EINVAL      : Invalid argument or mandatory field missing,
 *    unsupported indata kind (i.e. AAD CAAM_EE_WRAP_INDATA_AAD*),
 *    invalid tag, iv length,
 *    flags which need to be identical for each step of a multi-step operation
 *    have changed unexpectedly,
 *    update or finalize is called without previous init call,
 *    tag_len in final call does not match specified desired_tag_len
 * -EOVERFLOW   : Total length can't be represented with given iv length,
 *    in or out data of one block > 65535,
 *    out buffer can't store wrapped/unwrapped in buffer,
 *    the sum of the input data length of all requests exceeds the
 *    total_length provided in the first request
 * -ESTRPIPE    : Mismatch of total_len and sum of the data length of all
 *                requests
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBADMSG     : Authentication failed
 * -EOPNOTSUPP  : Length of session key is not supported
 * -EBUSY       : CAAM queue is full or
 *                previous operation on this session has not been completed
 * -EIO         : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EDEADLK     : Unexpected error in double buffer handling
 */
#define CAAM_WRAP_DATA \
	_IOWR('l', 1, struct caam_ee_wrap_unwrap_params)
#define CAAM_UNWRAP_DATA \
	_IOWR('l', 2, struct caam_ee_wrap_unwrap_params)

/*
 * encrypt decrypt ioctls
 * Encrypt or decrypt a (part of a) message
 * All but the last ioctl of each msg (CAAM_OP_SINGLE, CAAM_OP_MULTI_FINALIZE)
 * must provide data lengths aligned to the block_size.
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is not set the
 * encrypted/decrypted  output of the request is returned in the same request
 * (i.e. driver waits for CAAM to finish processing)
 * For each request the out_len has to be sufficient to store the output data
 * of this request.
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is set this is not true.
 * The ioctl of a multi request processing is left after scheduling the request.
 * This allows user-space to provide the second request while the first is still
 * processed. As soon as second request is scheduled the driver waits for the
 * first request to complete and returns the output data. Afterwards the
 * user-space can proceed with the third request. Only in the last request
 * (i.e. CAAM_OP_MULTI_FINALIZE) the ioctl waits for both requests to finish.
 * Therefore the out_len has to be sufficient to hold the data of the last and
 * the second to last request. For all other requests the out_len has to be
 * sufficient to store data of the preceding request. For the first request
 * the out_len might even be 0.
 * On successful completion of the ioctl the returned out_len is the actual
 * amount of bytes written to the out buffer.
 * The easiest way to handle interleaved requests is to use a buffer sufficient
 * to store the complete output data in user-space and provide the next-data
 * pointer via out and the remaining length in bytes in out_len.
 * On each successful request the out pointer needs to be incremented by the
 * returned out_len, while the remaining length is decremented accordingly.
 * Unless CAAM_EE_FLAG_NO_CLEAR_BIT is specified the confidential temporary
 * data (i.e. input data for encrypt, output data for decrypt) is overwritten in
 * kernel memory after each ioctl. In case the bit is specified corresponding
 * data buffers are only overwritten after the final call or in case of abort.
 *
 * Corresponding data structure : struct caam_ee_encrypt_decrypt_params
 *
 * Flags :
 *   Operation flags:
 *     CAAM_OP_SINGLE
 *     CAAM_OP_MULTI_INIT
 *     CAAM_OP_MULTI_UPDATE
 *     CAAM_OP_MULTI_FINALIZE
 *   Algorithm:
 *     CAAM_EE_CIPHER_AES_FLAG
 *     CAAM_EE_CIPHER_DES_FLAG
 *     CAAM_EE_CIPHER_3DES_FLAG
 *   Block-mode:
 *     CAAM_EE_ENCRYPT_BLOCK_ECB_FLAG
 *     CAAM_EE_ENCRYPT_BLOCK_CTR_FLAG
 *     CAAM_EE_ENCRYPT_BLOCK_CBC_FLAG
 *     CAAM_EE_ENCRYPT_BLOCK_CFB_FLAG
 *     CAAM_EE_ENCRYPT_BLOCK_OFB_FLAG
 *   Additional :
 *     CAAM_EE_FLAG_NO_CLEAR_BIT - skip overwriting confidential data
 *     CAAM_EE_FLAG_INTERLEAVED_OP_BIT - activate interleaved operation
 *
 * Error codes:
 * -ENOKEY      : No key set for current session
 * -EINVAL      : Invalid argument or mandatory field missing,
 *    invalid iv length,
 *    flags which need to be identical for each step of a multi-step operation
 *    have changed unexpectedly,
 *    update or finalize is called without previous init call
 * -EOVERFLOW   : In or out data of one block > 65535,
 *    out buffer can't store encrypted/decrypted in buffer
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EOPNOTSUPP  : Length of session key is not supported
 * -EBUSY       : CAAM queue is full or
 *                previous operation on this session has not been completed
 * -EIO         : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EDEADLK     : Unexpected error in double buffer handling
 */
#define CAAM_ENCRYPT_DATA \
	_IOWR('l', 3, struct caam_ee_encrypt_decrypt_params)
#define CAAM_DECRYPT_DATA \
	_IOWR('l', 4, struct caam_ee_encrypt_decrypt_params)


/*
 * sign verify ioctls
 * Sign or verify a (part of a) message
 * All but the last ioctl of each msg (CAAM_OP_SINGLE, CAAM_OP_MULTI_FINALIZE)
 * must provide data lengths aligned to the block_size.
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is not set the
 * sign/verify ioctls will only return after the input data of the request has
 * been processed (i.e. driver waits for CAAM to finish processing)
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is set this is not true.
 * The ioctl of a multi request processing is left after scheduling the request.
 * This allows user-space to provide the second request while the first is still
 * processed. As soon as second request is scheduled the driver waits for the
 * first request to complete. Afterwards the user-space can proceed with the
 * third request. Only in the last request (i.e. CAAM_OP_MULTI_FINALIZE)
 * the ioctl waits for both requests to finish.
 *
 * Corresponding data structure : struct caam_ee_sign_verify_params
 *
 * Flags :
 *   Operation flags:
 *     CAAM_OP_SINGLE
 *     CAAM_OP_MULTI_INIT
 *     CAAM_OP_MULTI_UPDATE
 *     CAAM_OP_MULTI_FINALIZE
 *   Algorithm:
 *     CAAM_EE_SIGN_ALG_HMAC_FLAG
 *     CAAM_EE_SIGN_ALG_RSA_PKCS1V15_FLAG
 *   Hash:
 *     CAAM_EE_HASH_MD5_FLAG
 *     CAAM_EE_HASH_SHA1_FLAG
 *     CAAM_EE_HASH_SHA224_FLAG
 *     CAAM_EE_HASH_SHA256_FLAG
 *   Additional :
 *     CAAM_EE_FLAG_INTERLEAVED_OP_BIT - activate interleaved operation
 *
 * Error codes:
 * -ENOKEY      : No key set for current session
 * -EINVAL      : Invalid argument or mandatory field missing,
 *    flags which need to be identical for each step of a multi-step operation
 *    have changed unexpectedly,
 *    update or finalize is called without previous init call
 * -EOVERFLOW   : In or out data of one block > 65535
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EOPNOTSUPP  : Length of session key is not supported
 * -EBUSY       : CAAM queue is full or
 *                previous operation on this session has not been completed
 * -EIO	        : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EDEADLK     : Unexpected error in double buffer handling
 */
#define CAAM_SIGN_DATA \
	_IOWR('l', 5, struct caam_ee_sign_verify_params)
#define CAAM_VERIFY_DATA \
	_IOW('l', 6, struct caam_ee_sign_verify_params)


/*
 * rng ioctl
 * Generate random data from DRNG or reseed DRNG
 *
 * Flags CAAM_EE_RNG_FLAG_RESEED and CAAM_EE_RNG_FLAG_GETRNG
 * are used to specify if reseed, random data generation, none or both
 * shall be done.
 *
 * For reseed the user might optionally specify
 * CAAM_EE_RNG_RESEED_LEN bytes of reseed data using reseed_buf.
 *
 * The length of data which can be generated in one call is limited
 * to CAAM_EE_MAX_RANDOM_LEN
 *
 * Corresponding data structure : struct caam_ee_rng_params
 *
 * Error codes:
 * -EINVAL      : Invalid flags
 *                length of requested data exceeds CAAM_EE_MAX_RANDOM_LEN
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBUSY       : CAAM queue is full or
 *                previous operation on this session has not been completed
 * -EIO	        : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EDEADLK     : Unexpected error in double buffer handling
 */
#define CAAM_RNG \
	_IOWR('l', 7, struct caam_ee_rng_params)


/*
 * abort the currently ongoing operation for this session
 *
 * Error codes:
 * -EINVAL	: no operation ongoing
 */
#define CAAM_ABORT_OPERATION \
	_IO('l', 9)

/*
 * Assign a keystore key specified by key identifier to the session
 *
 * Corresponding data structure : struct caam_ee_session_keystore_key_data
 *
 * Error codes:
 * -ENOKEY      : The key identifier has not been found in the key storage
 * -EACCES      : Neither effective user nor any group has the required
 *                permissions to use the key for any operation
 * -EOVERFLOW   : The key has already been referenced to frequently
 * -EBUSY       : Previous operation on this session has not been completed
 * -EAGAIN      : if caam_km device is not ready
 *                if keystore key is not activated
 *                if keystore is locked
 * -EFAULT      : Accessing ioctl data structure failed
 * -EINVAL	: DT link to caam_km is missing, invalid parameter
 * -ENOMEM	: Failed to allocate memory for red key decapsulation
 */
#define CAAM_SET_SESSION_KEYSTORE_KEY \
	_IOW('l', 10, struct caam_ee_session_keystore_key_data)

/*
 * Unset the key of the current session
 *
 * Error codes:
 * -ENOKEY	: no key set in session
 * -EBUSY	: previous operation on this session has not been completed
 */
#define CAAM_UNSET_SESSION_KEY \
	_IO('l', 11)


/*
 * Generate a builtin key using device specific persistent secret and
 * parameters and assign it to the session.
 *
 * Corresponding data structure : struct caam_ee_session_builtin_key_data
 *
 * The following key_flags are fixed
 *      CAAM_KEY_FLAG_STOR_BLACK
 *      CAAM_KEY_FLAG_TYPE_BUILTIN
 *      CAAM_KEY_FLAG_SUBTYPE_DERIVED
 *
 * CAAM_KEY_FLAG_SECRET_BYTES_... can be BYTES_PLAIN or BYTES_PARITY
 * CAAM_KEY_FLAG_AUTH_... can be UNKNOWN, LOW, MEDIUM, HIGH.
 * CAAM_KEY_FLAG_CONFIDENT_... can be UNKNOWN, LOW, MEDIUM, HIGH.
 *
 * If CAAM_KEY_FLAG_MODIFIER_NON is part of the spec.key_flags
 * spec.modifier == NULL and spec.modifier_bytes == 0
 * If CAAM_KEY_FLAG_MODIFIER_PUBLIC or CAAM_KEY_FLAG_MODIFIER_PRIVATE
 * is part of spec.key_flags spec.modifier != NULL and spec.modifier_bytes != 0
 * The userspace can set any value for bits of CAAM_KEY_FLAG_USER_MASK
 *
 * All other bits of spec.key_flags need to be unset.
 *
 * The length of the modifier is limited by CAAM_KEY_MOD_LEN_MAX
 *
 * Error codes:
 * -EACCES      : Neither effective user nor any group nor others have the
 *                required permissions to use the key for any operation
 * -ENOMEM      : Not sufficient memory to create builtin key
 * -EBUSY       : Previous operation on this session has not been completed
 *                or CAAM queue full
 * -EIO         : CAAM internal error
 * -EAGAIN      : if caam_km device is not ready
 * -EFAULT      : Accessing ioctl data structure failed
 * -EINVAL	: - DT link to caam_km is missing, invalid parameter
 *                - Invalid combination of spec.key_flags,
 *                  spec.key_bits, spec.perms,
 *                  spec.modifier, spec.modifier_bytes.
 *                - if key permissions exceed max allowed range
 * -EOPNOTSUPP  : in case length is not supported for key secret type
 */
#define CAAM_SET_SESSION_BUILTIN_KEY \
	_IOW('l', 12, struct caam_ee_session_builtin_key_data)

/*
 * get info of currently loaded key
 *
 * Error codes:
 * -ENOMEM      : The memory for modifier or aad is not suffient to
 *                store the data - in this case the length is
 *                set to return the required length
 * -EFAULT      : Accessing ioctl data structure failed
 */
#define CAAM_GET_SESSION_KEY_INFO \
	_IOWR('l', 14, struct caam_key_info)

/*
 * dgst ioctl
 * Calculate the digest of (part of a) message
 * All but the last ioctl of each msg (CAAM_OP_SINGLE, CAAM_OP_MULTI_FINALIZE)
 * must provide data lengths aligned to the block_size.
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is not set the
 * dgst ioctl will only return after the input data of the request has
 * been processed (i.e. driver waits for CAAM to finish processing)
 * If flag CAAM_EE_FLAG_INTERLEAVED_OP_BIT is set this is not true.
 * The ioctl of a multi request processing is left after scheduling the request.
 * This allows user-space to provide the second request while the first is still
 * processed. As soon as second request is scheduled the driver waits for the
 * first request to complete. Afterwards the user-space can proceed with the
 * third request. Only in the last request (i.e. CAAM_OP_MULTI_FINALIZE)
 * the ioctl waits for both requests to finish.
 *
 * Corresponding data structure : struct caam_ee_dgst_params
 *
 * Flags :
 *   Operation flags:
 *     CAAM_OP_SINGLE
 *     CAAM_OP_MULTI_INIT
 *     CAAM_OP_MULTI_UPDATE
 *     CAAM_OP_MULTI_FINALIZE
 *   Hash:
 *     CAAM_EE_HASH_MD5_FLAG
 *     CAAM_EE_HASH_SHA1_FLAG
 *     CAAM_EE_HASH_SHA224_FLAG
 *     CAAM_EE_HASH_SHA256_FLAG
 *   Additional :
 *     CAAM_EE_FLAG_INTERLEAVED_OP_BIT - activate interleaved operation
 *
 * Error codes:
 * -EINVAL      : Invalid argument or mandatory field missing,
 *    flags which need to be identical for each step of a multi-step operation
 *    have changed unexpectedly,
 *    update or finalize is called without previous init call
 * -EOVERFLOW   : In or out data of one block > 65535
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBUSY       : CAAM queue is full or
 *                previous operation on this session has not been completed
 * -EIO	        : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EDEADLK     : Unexpected error in double buffer handling
 */
#define CAAM_DGST_DATA \
	_IOWR('l', 15, struct caam_ee_dgst_params)

/*
 * Importing a wrapped key(K1) into keystorage needs to be handled in
 * two stages.
 * First of all the wrapped keysecret needs to be unwrapped.
 * Afterwards it needs to be imported (installed) to the keystorage.
 *
 * The first step needs to be handled by caam_ee using the key(K2) currently
 * attached to the caam_ee session. This is required in order to perform
 * user/(supplementary)groups/permission checks.
 * In order to be allowed to unwrap a wrapped key(K1) either
 * CAAM_PERM_KEY_UNWRAP_BMSK or CAAM_PERM_DATA_UNWRAP_BMSK is required.
 * This step will provide an the encapsulated key(K3) which can be used
 * in the second step.
 *
 * In the second step the encapsulated key(K3) is imported using
 * CAAM_KM_INSTALL_KEYSTORE_KEY with CAAM_KEY_FLAG_SUBTYPE_IMPORTED.
 *
 * Beside the encapsulated key(K3) itself the output will include
 * struct caam_key_encap_info at the very beginning.
 * This struct is used by CAAM_KM_INSTALL_KEYSTORE_KEY to
 * check that the uid is valid, permissions don't exceed the
 * limit defined by the inheritance_mask of the key(K2) used to unwrap
 * the wrapped key(K1), authenticity and confidentiality limits
 * defined by the key(K2) used to unwrap the wrapped key(K1).
 *
 * The algorithm and block mode of wrapped keys is fixed to AES_CCM.
 *
 * Corresponding data structure : struct caam_ee_wrapped_key_to_blob
 *
 * Error codes:
 * -ENOKEY      : No key set for current session
 * -EINVAL      : Invalid argument or mandatory field missing,
 *                invalid tag, iv length
 * -EOVERFLOW   : Total length can't be represented with given iv length,
 *                in or out data of one block > 65535,
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBADMSG     : Authentication failed
 * -EOPNOTSUPP  : Length of session key is not supported
 * -EBUSY       : CAAM queue is full or
 *                previous operation on this session has not been completed
 * -EIO         : CAAM internal error
 * -EFAULT      : Accessing ioctl data structure failed
 * -EDEADLK     : Unexpected error in double buffer handling
 */
#define CAAM_EE_PREPARE_WRAPPED_KEY_IMPORT \
	_IOWR('l', 20, struct caam_ee_prep_wrapped_key_import)


/*
 * asymmetric encrypt decrypt ioctls
 * Asymmetric RSA encrypt or decrypt of a message
 *
 * In contrast to CAAM_ENCRYPT_DATA and CAAM_DECRYPT_DATA the length of the
 * message is short - limited by the length of the asymmetric key and
 * the selected padding.
 * Therefore all data needs to be encrypted or decrypted in a single call.
 *
 * The cipher length needs to be always equal to the length of the used key.
 * In case CAAM_EE_ASYM_ENC_DEC_PAD_PKCS1_V15 is set PKCS1 V1.5 padding
 * is added or removed. The plain length needs to be less or equal to
 * length of key - 11(CAAM_EE_RSA_ENC_PKCS1_V15_PAD_OVERHEAD) in this case.
 * In case CAAM_EE_ASYM_ENC_DEC_PAD_RAW is set no padding is used. In this
 * case plain length needs to be equal to the length of the used key.
 *
 * Corresponding data structure : struct caam_ee_asym_encrypt_decrypt_params
 *
 * Flags :
 *   Padding:
 *     CAAM_EE_ASYM_ENC_DEC_PAD_RAW
 *     CAAM_EE_ASYM_ENC_DEC_PAD_PKCS1_V15
 *
 * Error codes:
 * -ENOKEY      : No key set for current session
 * -EINVAL      : Invalid argument (e.g. input len), mandatory field missing
 * -EOVERFLOW   : Output buffer is too small to store the result
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBUSY       : Previous operation on this session has not been completed
 * -EOPNOTSUPP  : Key is not supported (i.e. symmetric key loaded)
 * -EACCESS     : Current user/grp has not the required rights to use the key
 * -EPERM       : Trying to perform decrypt with a public key
 * -EFAULT      : Accessing ioctl data structure failed
 */
#define CAAM_ASYM_ENCRYPT_DATA \
	_IOWR('l', 30, struct caam_ee_asym_encrypt_decrypt_params)
#define CAAM_ASYM_DECRYPT_DATA \
	_IOWR('l', 31, struct caam_ee_asym_encrypt_decrypt_params)

/*
 * asymmetric sign/verify ioctls
 * Generate/Verify an RSA signature of an externally provided hash
 *
 * Note: Generation/Verification of RSA signatures for a complete message
 * is handled by CAAM_SIGN_DATA/CAAM_VERIFY_DATA ioctls.
 * In that case the hash is computed internally using CAAM.
 *
 * As the hash length is short, all processing is handled in a single call.
 *
 * The hash length needs to be always equal to the max length of the used
 * hash algorithm.
 * For CAAM_ASYM_VERIFY_HASH the provided signature buffer needs to be
 * exactly the length of the used key.
 * For CAAM_ASYM_SIGN_HASH the provided signature buffer needs to be as
 * least as big as the used key.
 *
 * Corresponding data structure : struct caam_ee_asym_sign_verify_hash_params
 *
 * Flags :
 *   Hash:
 *     CAAM_EE_HASH_MD5_FLAG
 *     CAAM_EE_HASH_SHA1_FLAG
 *     CAAM_EE_HASH_SHA224_FLAG
 *     CAAM_EE_HASH_SHA256_FLAG
 *
 * Error codes:
 * -ENOKEY      : No key set for current session
 * -EINVAL      : Invalid argument (e.g. input len), mandatory field missing
 * -EOVERFLOW   : Signature output buffer of CAAM_ASYM_SIGN_HASH is too small
 * -ENOMEM      : No memory to allocate temporary buffers in kernel
 * -EBUSY       : Previous operation on this session has not been completed
 * -EOPNOTSUPP  : Key is not supported (i.e. symmetric key loaded)
 * -EACCESS     : Current user/grp has not the required rights to use the key
 * -EPERM       : Trying to perform sign with a public key
 * -EFAULT      : Accessing ioctl data structure failed
 */
#define CAAM_ASYM_SIGN_HASH \
	_IOWR('l', 40, struct caam_ee_asym_sign_verify_hash_params)
#define CAAM_ASYM_VERIFY_HASH \
	_IOWR('l', 41, struct caam_ee_asym_sign_verify_hash_params)

/* caam_ee IOCTL flags common for sign/verify, encrypt/decrypt, wrap/unwrap */
#define CAAM_OP_SHIFT		0
#define CAAM_OP_MASK		0x03
#define CAAM_OP_SINGLE		(0 << CAAM_OP_SHIFT)
#define CAAM_OP_MULTI_INIT	(1 << CAAM_OP_SHIFT)
#define CAAM_OP_MULTI_UPDATE	(2 << CAAM_OP_SHIFT)
#define CAAM_OP_MULTI_FINALIZE	(3 << CAAM_OP_SHIFT)

#define CAAM_EE_FLAG_NO_CLEAR_SHIFT 16
#define CAAM_EE_FLAG_NO_CLEAR_BIT (1<<CAAM_EE_FLAG_NO_CLEAR_SHIFT)
#define CAAM_EE_FLAG_NO_CLEAR_MASK CAAM_EE_FLAG_NO_CLEAR_BIT

#define CAAM_EE_FLAG_INTERLEAVED_OP_SHIFT 17
#define CAAM_EE_FLAG_INTERLEAVED_OP_BIT \
	(1<<CAAM_EE_FLAG_INTERLEAVED_OP_SHIFT)
#define CAAM_EE_FLAG_INTERLEAVED_OP_MASK \
	CAAM_EE_FLAG_INTERLEAVED_OP_BIT



/* caam_ee IOCTL flag mappings for wrap, unwrap, encrypt, decrypt */
#define CAAM_EE_CIPHER_SHIFT	4
#define CAAM_EE_CIPHER_MASK	0xF0
/* index (see caam_ee_cipher_descs) */
#define CAAM_EE_CIPHER_AES_IDX	0
#define CAAM_EE_CIPHER_DES_IDX	1
#define CAAM_EE_CIPHER_3DES_IDX	2
/* values as used in flags */
#define CAAM_EE_CIPHER_AES_FLAG	\
	(CAAM_EE_CIPHER_AES_IDX << CAAM_EE_CIPHER_SHIFT)
#define CAAM_EE_CIPHER_DES_FLAG	\
	(CAAM_EE_CIPHER_DES_IDX << CAAM_EE_CIPHER_SHIFT)
#define CAAM_EE_CIPHER_3DES_FLAG \
	(CAAM_EE_CIPHER_3DES_IDX << CAAM_EE_CIPHER_SHIFT)
/* bitmasks */
#define CAAM_EE_CIPHER_AES_BMSK	  (1 << CAAM_EE_CIPHER_AES_IDX)
#define CAAM_EE_CIPHER_DES_BMSK	  (1 << CAAM_EE_CIPHER_DES_IDX)
#define CAAM_EE_CIPHER_3DES_BMSK  (1 << CAAM_EE_CIPHER_3DES_IDX)


/* caam_ee IOCTL flags,etc. common for encrypt/decrypt */
#define CAAM_EE_ENCRYPT_BLOCK_SHIFT		8
#define CAAM_EE_ENCRYPT_BLOCK_MASK		0xF00
/* index (see caam_ee_wrap_block_modes_descs) */
#define CAAM_EE_ENCRYPT_BLOCK_ECB_IDX		0
#define CAAM_EE_ENCRYPT_BLOCK_CTR_IDX		1
#define CAAM_EE_ENCRYPT_BLOCK_CBC_IDX		2
#define CAAM_EE_ENCRYPT_BLOCK_CFB_IDX		3
#define CAAM_EE_ENCRYPT_BLOCK_OFB_IDX		4
/* values as used in flags */
#define CAAM_EE_ENCRYPT_BLOCK_ECB_FLAG \
	(CAAM_EE_ENCRYPT_BLOCK_ECB_IDX << CAAM_EE_ENCRYPT_BLOCK_SHIFT)
#define CAAM_EE_ENCRYPT_BLOCK_CTR_FLAG \
	(CAAM_EE_ENCRYPT_BLOCK_CTR_IDX << CAAM_EE_ENCRYPT_BLOCK_SHIFT)
#define CAAM_EE_ENCRYPT_BLOCK_CBC_FLAG \
	(CAAM_EE_ENCRYPT_BLOCK_CBC_IDX << CAAM_EE_ENCRYPT_BLOCK_SHIFT)
#define CAAM_EE_ENCRYPT_BLOCK_CFB_FLAG \
	(CAAM_EE_ENCRYPT_BLOCK_CFB_IDX << CAAM_EE_ENCRYPT_BLOCK_SHIFT)
#define CAAM_EE_ENCRYPT_BLOCK_OFB_FLAG \
	(CAAM_EE_ENCRYPT_BLOCK_OFB_IDX << CAAM_EE_ENCRYPT_BLOCK_SHIFT)
/* bitmasks */
#define CAAM_EE_ENCRYPT_BLOCK_ECB_BMSK	(1 << CAAM_EE_ENCRYPT_BLOCK_ECB_IDX)
#define CAAM_EE_ENCRYPT_BLOCK_CTR_BMSK	(1 << CAAM_EE_ENCRYPT_BLOCK_CTR_IDX)
#define CAAM_EE_ENCRYPT_BLOCK_CBC_BMSK	(1 << CAAM_EE_ENCRYPT_BLOCK_CBC_IDX)
#define CAAM_EE_ENCRYPT_BLOCK_CFB_BMSK	(1 << CAAM_EE_ENCRYPT_BLOCK_CFB_IDX)
#define CAAM_EE_ENCRYPT_BLOCK_OFB_BMSK	(1 << CAAM_EE_ENCRYPT_BLOCK_OFB_IDX)

/* overall masks for encrypt/decrypt flags */
#define CAAM_EE_ENCRYPT_DECRYPT_OP_MASK_STATE_INDEP (CAAM_EE_CIPHER_MASK |\
		CAAM_EE_ENCRYPT_BLOCK_MASK |\
		CAAM_EE_FLAG_NO_CLEAR_MASK |\
		CAAM_EE_FLAG_INTERLEAVED_OP_MASK)
#define CAAM_EE_ENCRYPT_DECRYPT_OP_MASK (CAAM_OP_MASK |\
		CAAM_EE_ENCRYPT_DECRYPT_OP_MASK_STATE_INDEP)

/* structure for encrypt or decrypt ioctls */
struct caam_ee_encrypt_decrypt_params {
	/* pointer and length of input data (input) */
	uint8_t *in;
	uint32_t in_len;

	/* pointer and length of output data (output) */
	uint8_t *out;
	uint32_t out_len;

	/*
	 * pointer and length of initialization vector (input)
	 * only needed for CAAM_OP_SINGLE and CAAM_OP_MULTI_INIT
	 */
	uint8_t *iv;
	uint32_t iv_len;

	/* operation flags (input) */
	uint32_t flags;
};


/* caam_ee IOCTL flags,etc. common for wrap/unwrap */
#define CAAM_EE_WRAP_BLOCK_SHIFT		8
#define CAAM_EE_WRAP_BLOCK_MASK			0xF00
/* index (see caam_ee_wrap_block_modes_descs) */
#define CAAM_EE_WRAP_BLOCK_CCM_IDX		0
/* values as used in flags */
#define CAAM_EE_WRAP_BLOCK_CCM_FLAG \
	(CAAM_EE_WRAP_BLOCK_CCM_IDX << CAAM_EE_WRAP_BLOCK_SHIFT)
/* bitmasks */
#define CAAM_EE_WRAP_BLOCK_CCM_BMSK	(1 << CAAM_EE_WRAP_BLOCK_CCM_IDX)

/* flags, bmsks for input data specification */
#define CAAM_EE_WRAP_INDATA_SHIFT		12
#define CAAM_EE_WRAP_INDATA_MASK		0xF000
#define CAAM_EE_WRAP_INDATA_NORM_IDX		0
#define CAAM_EE_WRAP_INDATA_AAD_IDX		1
#define CAAM_EE_WRAP_INDATA_NORM_FLAG \
	(CAAM_EE_WRAP_INDATA_NORM_IDX << CAAM_EE_WRAP_INDATA_SHIFT)
#define CAAM_EE_WRAP_INDATA_AAD_FLAG \
	(CAAM_EE_WRAP_INDATA_AAD_IDX << CAAM_EE_WRAP_INDATA_SHIFT)

/* overall masks for wrap/unwrap flags */
#define CAAM_EE_WRAP_UNWRAP_OP_MASK_STATE_INDEP (CAAM_EE_CIPHER_MASK |\
		CAAM_EE_WRAP_BLOCK_MASK | CAAM_EE_WRAP_INDATA_MASK |\
		CAAM_EE_FLAG_NO_CLEAR_MASK |\
		CAAM_EE_FLAG_INTERLEAVED_OP_MASK)
#define CAAM_EE_WRAP_UNWRAP_OP_MASK (CAAM_OP_MASK |\
		CAAM_EE_WRAP_UNWRAP_OP_MASK_STATE_INDEP)

/* structure for wrap or unwrap ioctls */
struct caam_ee_wrap_unwrap_params {
	/* pointer and length of input data (input) */
	uint8_t *in;
	uint32_t in_len;

	/* pointer and length of output data (output) */
	uint8_t *out;
	uint32_t out_len;

	/*
	 * pointer and length of initialization vector (input)
	 * only needed for CAAM_OP_SINGLE and CAAM_OP_MULTI_INIT
	 */
	uint8_t *iv;
	uint32_t iv_len;

	/*
	 * pointer and length of tag (input, output)
	 * only needed for CAAM_OP_SINGLE and CAAM_OP_MULTI_FINALIZE
	 */
	uint8_t *tag;
	uint32_t tag_len;

	/*
	 * total length of the data (input)
	 * only needed for CAAM_OP_SINGLE and CAAM_OP_MULTI_INIT
	 */
	uint32_t total_len;

	/*
	 * desired tag length of the data (input)
	 * only needed for CAAM_OP_SINGLE and CAAM_OP_MULTI_INIT
	 */
	uint32_t desired_tag_len;

	/* operation flags (input) */
	uint32_t flags;
};


/* caam_ee IOCTL flag mappings for sign, verify */
#define CAAM_EE_SIGN_ALG_SHIFT			4
#define CAAM_EE_SIGN_ALG_MASK			0xF0
/* index (see caam_ee_sign_alg_descs) */
#define CAAM_EE_SIGN_ALG_HMAC_IDX			0
#define CAAM_EE_SIGN_ALG_RSA_PKCS1V15_IDX		1
/* values as used in flags */
#define CAAM_EE_SIGN_ALG_HMAC_FLAG	\
	(CAAM_EE_SIGN_ALG_HMAC_IDX << CAAM_EE_SIGN_ALG_SHIFT)
#define CAAM_EE_SIGN_ALG_RSA_PKCS1V15_FLAG	\
	(CAAM_EE_SIGN_ALG_RSA_PKCS1V15_IDX << CAAM_EE_SIGN_ALG_SHIFT)
/* bitmasks */
#define CAAM_EE_SIGN_ALG_HMAC_BMSK		\
	(1 << CAAM_EE_SIGN_ALG_HMAC_IDX)
#define CAAM_EE_SIGN_ALG_RSA_PKCS1V15_BMSK	\
	(1 << CAAM_EE_SIGN_ALG_RSA_PKCS1V15_IDX)

#define CAAM_EE_HASH_SHIFT			8
#define CAAM_EE_HASH_MASK			0xF00
/* index (see caam_ee_hash_modes_descs) */
#define CAAM_EE_HASH_MD5_IDX		0
#define CAAM_EE_HASH_SHA1_IDX		1
#define CAAM_EE_HASH_SHA224_IDX		2
#define CAAM_EE_HASH_SHA256_IDX		3
/* values as used in flags */
#define CAAM_EE_HASH_MD5_FLAG \
	(CAAM_EE_HASH_MD5_IDX << CAAM_EE_HASH_SHIFT)
#define CAAM_EE_HASH_SHA1_FLAG \
	(CAAM_EE_HASH_SHA1_IDX << CAAM_EE_HASH_SHIFT)
#define CAAM_EE_HASH_SHA224_FLAG \
	(CAAM_EE_HASH_SHA224_IDX << CAAM_EE_HASH_SHIFT)
#define CAAM_EE_HASH_SHA256_FLAG \
	(CAAM_EE_HASH_SHA256_IDX << CAAM_EE_HASH_SHIFT)
/* bitmasks */
#define CAAM_EE_HASH_MD5_BMSK		(1 << CAAM_EE_HASH_MD5_IDX)
#define CAAM_EE_HASH_SHA1_BMSK		(1 << CAAM_EE_HASH_SHA1_IDX)
#define CAAM_EE_HASH_SHA224_BMSK	(1 << CAAM_EE_HASH_SHA224_IDX)
#define CAAM_EE_HASH_SHA256_BMSK	(1 << CAAM_EE_HASH_SHA256_IDX)

/* overall masks for sign/verify flags */
#define CAAM_EE_SIGN_VERIFY_OP_MASK_STATE_INDEP (CAAM_EE_SIGN_ALG_MASK |\
		CAAM_EE_HASH_MASK |\
		CAAM_EE_FLAG_INTERLEAVED_OP_MASK)
#define CAAM_EE_SIGN_VERIFY_OP_MASK (CAAM_OP_MASK |\
		CAAM_EE_SIGN_VERIFY_OP_MASK_STATE_INDEP)

/* structure for sign or verify ioctls */
struct caam_ee_sign_verify_params {
	/* pointer and length of input data (input) */
	uint8_t *in;
	uint32_t in_len;

	/*
	 * pointer and length of tag (input, output)
	 * only needed for CAAM_OP_SINGLE and CAAM_OP_MULTI_FINALIZE
	 */
	uint8_t *tag;
	uint32_t tag_len;

	/* operation flags (input) */
	uint32_t flags;
};

/* overall masks for dgst flags */
#define CAAM_EE_DGST_OP_MASK_STATE_INDEP (CAAM_EE_HASH_MASK |\
		CAAM_EE_FLAG_INTERLEAVED_OP_MASK)
#define CAAM_EE_DGST_OP_MASK (CAAM_OP_MASK |\
		CAAM_EE_DGST_OP_MASK_STATE_INDEP)

/* structure for dgst ioctl */
struct caam_ee_dgst_params {
	/* pointer and length of input data (input) */
	uint8_t *in;
	uint32_t in_len;

	/*
	 * pointer and length of digest (input, output)
	 * only needed for CAAM_OP_SINGLE and CAAM_OP_MULTI_FINALIZE
	 */
	uint8_t *dgst;
	uint32_t dgst_len;

	/* operation flags (input) */
	uint32_t flags;
};


#define CAAM_EE_RNG_FLAG_RESEED 1
#define CAAM_EE_RNG_FLAG_GETRNG 2
#define CAAM_EE_RNG_FLAGS (CAAM_EE_RNG_FLAG_RESEED | CAAM_EE_RNG_FLAG_GETRNG)

#define CAAM_EE_RNG_RESEED_LEN 32
#define CAAM_EE_MAX_RANDOM_LEN CAAM_EE_MAX_DATA_LEN

struct caam_ee_rng_params {
	/* in case not NULL length has to be CAAM_EE_RNG_RESEED_LEN */
	uint8_t *reseed_buf;

	uint8_t *random_out;
	uint32_t random_out_len;

	uint32_t rng_flags;
};


/* struct for IOCTL CAAM_SET_SESSION_KEYSTORE_KEY */
struct caam_ee_session_keystore_key_data {
	uint32_t key_id;
};

#define CAAM_EE_BUILTIN_KEY_MOD_LEN_MAX CAAM_KEY_BUILTIN_KEY_MOD_LEN_MAX

/* struct for IOCTL CAAM_SET_SESSION_BUILTIN_KEY */
struct caam_ee_session_builtin_key_data {
	struct caam_key_spec spec;
};

/* structure of CAAM_EE_PREPARE_WRAPPED_KEY_IMPORT IOCTL */
struct caam_ee_prep_wrapped_key_import {
	/* reserved for future use - needs to be 0 at the moment */
	uint32_t flags;

	/* pointer and length of input data (input) */
	uint8_t *wrapped_key;
	uint32_t wrapped_key_len;

	/* pointer and length of initialization vector (input) */
	uint8_t *wrapped_iv;
	uint32_t wrapped_iv_len;

	/* pointer and length of tag (input) */
	uint8_t *wrapped_tag;
	uint32_t wrapped_tag_len;

	/*
	 * encapsulated key output
	 * this output can be used for CAAM_KM_INSTALL_KEYSTORE_KEY
	 * CAAM_KEY_FLAG_SUBTYPE_IMPORTED
	 */
	uint8_t *key_import_data;
	/* inout - in: avail length for key_import_data - out: used length */
	uint32_t key_import_data_len;
};

/* operation flag masks for asymmetric encrypt/decrypt */
#define CAAM_EE_ASYM_ENC_DEC_PAD_MASK		0x0F
#define CAAM_EE_ASYM_ENC_DEC_PAD_RAW		0x00
#define CAAM_EE_ASYM_ENC_DEC_PAD_PKCS1_V15	0x01

#define CAAM_EE_ASYM_ENCRYPT_DECRYPT_OP_MASK \
		(CAAM_EE_ASYM_ENC_DEC_PAD_MASK)

/* structure for encrypt or decrypt ioctls */
struct caam_ee_asym_encrypt_decrypt_params {
	/* pointer and length of input data (input) */
	uint8_t *in;
	uint32_t in_len;

	/* pointer and length of output data (output) */
	uint8_t *out;
	uint32_t out_len;

	/* operation flags (input) */
	uint32_t flags;
};

/* some defines to compute the max data length */
#define CAAM_EE_RSA_ENC_PKCS1_V15_PAD_OVERHEAD	11
#define CAAM_EE_RSA_ENC_PKCS1_V15_MAX_PLAIN_LEN(keybytes) \
	(keybytes - CAAM_EE_RSA_ENC_PKCS1_V15_PAD_OVERHEAD)

/* operation flag masks for asymmetric sign/verify of hash */
#define CAAM_EE_ASYM_SIGN_VERIFY_HASH_OP_MASK \
		(CAAM_EE_HASH_MASK)

/* structure for sign or verify of hash ioctls */
struct caam_ee_asym_sign_verify_hash_params {
	/* pointer and length of hash (always input) */
	uint8_t *hash;
	uint32_t hash_len;

	/*
	 * pointer and length of asymmetric signature
	 * output for CAAM_ASYM_SIGN_HASH
	 * input for CAAM_ASYM_VERIFY_HASH
	 */
	uint8_t *signature;
	uint32_t signature_len;

	/* operation flags (input) */
	uint32_t flags;
};

/*
 * The below structure allows the user-space to decide if it is compatible
 * to the current kernel version.
 * In order to do so the user-space needs to know if new IOCTLs
 * have been added, IOCTLs have been removed or data structures have
 * been changed. Last but not least it needs to know if there are any
 * user-space relevant changes regarding the behavior of IOCTLs
 * (e.g. functionality of IOCTL has been implemented, extended)
 *
 * Whenever a completely new set of IOCTLs is provided the version number
 * needs to be incremented. But as the already existing IOCTLs are not
 * touched the ioctl_comp_version can stay the same.
 * subversion is set to 0 in this case
 *
 * In case an already existing IOCTL is changed (with regards to the
 * input parameters or removed) the version gets incremented and
 * ioctl_comp_version is set to the same value.
 * subversion is set to 0 in this case
 *
 * For minor changes. E.g. functionality of IOCTL implemented, extended
 * only subversion is incremented
 */

#define CAAM_EE_CURRENT_VERSION			12
#define CAAM_EE_CURRENT_SUBVERSION		0
#define CAAM_EE_CURRENT_IOCTL_COMP_VERSION	9

struct caam_ee_version {
	uint32_t version;
	uint32_t subversion;
	uint32_t ioctl_comp_version;
};



/* defines */
#define CAAM_EE_MAX_TOTAL_LEN			0xFFFFFFFF
#define CAAM_EE_MAX_DATA_LEN			0xFFFF

#define CAAM_EE_AES_BLOCK_SIZE			16
#define CAAM_EE_AES_CCM_IV_MOD			1
#define CAAM_EE_AES_CCM_IV_MIN			7
#define CAAM_EE_AES_CCM_IV_MAX			13
#define CAAM_EE_AES_CCM_TAG_MOD			2
#define CAAM_EE_AES_CCM_TAG_MIN			4
#define CAAM_EE_AES_CCM_TAG_MAX			16
#define CAAM_EE_AES_ECB_IV_MOD			1
#define CAAM_EE_AES_ECB_IV_MIN_MAX		0
#define CAAM_EE_AES_CTR_IV_MOD			1
#define CAAM_EE_AES_CTR_IV_MIN			12
#define CAAM_EE_AES_CTR_IV_MAX			16
#define CAAM_EE_AES_CBC_OFB_CFB_IV_MOD		1
#define CAAM_EE_AES_CBC_OFB_CFB_IV_MIN_MAX	16

#define CAAM_EE_DES_3DES_BLOCK_SIZE		8
#define CAAM_EE_DES_3DES_ECB_IV_MOD		1
#define CAAM_EE_DES_3DES_ECB_IV_MIN_MAX		0
#define CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MOD	1
#define CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MIN_MAX	8

/* HASH defines */
#define CAAM_EE_HASH_ALIGN		64
#define CAAM_EE_HASH_MD5_LEN_MAX	16
#define CAAM_EE_HASH_SHA1_LEN_MAX	20
#define CAAM_EE_HASH_SHA224_LEN_MAX	28
#define CAAM_EE_HASH_SHA256_LEN_MAX	32
/* Recommendated min (see RFC2104) */
#define CAAM_EE_HASH_MD5_LEN_MIN	(CAAM_EE_HASH_MD5_LEN_MAX/2)
#define CAAM_EE_HASH_SHA1_LEN_MIN	(CAAM_EE_HASH_SHA1_LEN_MAX/2)
#define CAAM_EE_HASH_SHA224_LEN_MIN	(CAAM_EE_HASH_SHA224_LEN_MAX/2)
#define CAAM_EE_HASH_SHA256_LEN_MIN	(CAAM_EE_HASH_SHA256_LEN_MAX/2)

/* key length supported by AES, DES, 3DES */
#define CAAM_EE_CIPHER_DES_KEY_LENGTH	(CAAM_KEY_LEN_56BIT_BMSK |\
		CAAM_KEY_LEN_64BIT_BMSK)
#define CAAM_EE_CIPHER_3DES_KEY_LENGTH	(CAAM_KEY_LEN_112BIT_BMSK |\
		CAAM_KEY_LEN_168BIT_BMSK | CAAM_KEY_LEN_128BIT_BMSK |\
		CAAM_KEY_LEN_192BIT_BMSK)
#define CAAM_EE_CIPHER_AES_KEY_LENGTH	(CAAM_KEY_LEN_128BIT_BMSK |\
		CAAM_KEY_LEN_192BIT_BMSK |\
		CAAM_KEY_LEN_256BIT_BMSK)


/*
 * define capabilities of ciphers and block modes, ioctls
 * in a way use-able by kernel and user-space
 * The idx of the arrays need to match "*_IDX" defined above
 */

/* ciphers supported by encrypt / decrypt */
#define CAAM_EE_ENCRYPT_CIPHERS	(CAAM_EE_CIPHER_AES_BMSK |\
		CAAM_EE_CIPHER_DES_BMSK |\
		CAAM_EE_CIPHER_3DES_BMSK)

/* ciphers supported by wrap unwrap */
#define CAAM_EE_WRAP_CIPHERS	(CAAM_EE_CIPHER_AES_BMSK)

/* block mode descriptions */
struct caam_ee_blk_desc {
	/* min iv length */
	uint32_t iv_min;
	/* max iv length */
	uint32_t iv_max;
	/* if > 1 iv has to fulfill % iv_mod == 0 */
	uint32_t iv_mod;
	/* min tag length */
	uint32_t tag_min;
	/* max tag length */
	uint32_t tag_max;
	/* if > 1 tag has to fulfill % tag_mod == 0 */
	uint32_t tag_mod;
	/* the complete message must not be longer than this value */
	uint32_t max_total_len;
	/* the maximum amount of data one IOCTL can process at once */
	uint32_t max_block_len;
	/*
	 * All intermediate data blocks will be a multiple of the block size.
	 * In case the below value is true the last data block needs to
	 * be aligned too
	 */
	bool final_block_aligned;
};

/* cipher descriptions */
struct caam_ee_cipher_desc {
	/* bitmask encoding the supported key lengths of algorithm */
	uint32_t key_len_msk;
	/* block size */
	uint32_t block_size;
	/* supported block modes for wrap unwrap */
	uint32_t wrap_blk_mode_msk;
	/*
	 * Pointer to corresponding wrap unwrap description
	 * This struct has to contain entries for all block modes set
	 * in wrap_blk_mode_msk
	 */
	const struct caam_ee_blk_desc *wrap_blk_descs;
	/* supported block modes for encrypt decrypt */
	uint32_t encrypt_blk_mode_msk;
	/*
	 * Pointer to corresponding encrypt decrypt description
	 * This struct has to contain entries for all block modes set
	 * in encrypt_blk_mode_msk
	 */
	const struct caam_ee_blk_desc *encrypt_blk_descs;
};



#define CAAM_EE_ENCRYPT_BLOCK_BMSKS_DES_3DES (CAAM_EE_ENCRYPT_BLOCK_ECB_BMSK |\
		CAAM_EE_ENCRYPT_BLOCK_CBC_BMSK |\
		CAAM_EE_ENCRYPT_BLOCK_CFB_BMSK |\
		CAAM_EE_ENCRYPT_BLOCK_OFB_BMSK)
#define CAAM_EE_ENCRYPT_BLOCK_BMSKS_AES (CAAM_EE_ENCRYPT_BLOCK_CTR_BMSK |\
		CAAM_EE_ENCRYPT_BLOCK_BMSKS_DES_3DES)

static const struct caam_ee_blk_desc caam_ee_wrap_blk_aes_descs[] = {
	/* CAAM_EE_WRAP_BLOCK_CCM_IDX */
	[CAAM_EE_WRAP_BLOCK_CCM_IDX] =
	{
		/* IV min,max,mod */
		CAAM_EE_AES_CCM_IV_MIN,
		CAAM_EE_AES_CCM_IV_MAX,
		CAAM_EE_AES_CCM_IV_MOD,
		/* tag min,max,mod */
		CAAM_EE_AES_CCM_TAG_MIN,
		CAAM_EE_AES_CCM_TAG_MAX,
		CAAM_EE_AES_CCM_TAG_MOD,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		false /* no padding for final block */
	}
};

static const struct caam_ee_blk_desc caam_ee_encrypt_blk_aes_descs[] = {
	/* CAAM_EE_ENCRYPT_BLOCK_ECB_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_ECB_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_AES_ECB_IV_MIN_MAX,
		CAAM_EE_AES_ECB_IV_MIN_MAX,
		CAAM_EE_AES_ECB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		true /* padding for final block */
	},

	/* CAAM_EE_ENCRYPT_BLOCK_CTR_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_CTR_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_AES_CTR_IV_MIN,
		CAAM_EE_AES_CTR_IV_MAX,
		CAAM_EE_AES_CTR_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		false /* no padding for final block */
	},

	/* CAAM_EE_ENCRYPT_BLOCK_CBC_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_CBC_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_AES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_AES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_AES_CBC_OFB_CFB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		true /* padding for final block */
	},

	/* CAAM_EE_ENCRYPT_BLOCK_CFB_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_CFB_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_AES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_AES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_AES_CBC_OFB_CFB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		true /* padding for final block */
	},

	/* CAAM_EE_ENCRYPT_BLOCK_OFB_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_OFB_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_AES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_AES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_AES_CBC_OFB_CFB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		false /* no padding for final block */
	}
};

static const struct caam_ee_blk_desc caam_ee_encrypt_blk_des_3des_descs[] = {
	/* CAAM_EE_ENCRYPT_BLOCK_ECB_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_ECB_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_DES_3DES_ECB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_ECB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_ECB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		true /* padding for final block */
	},

	/* CAAM_EE_ENCRYPT_BLOCK_CBC_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_CBC_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		true /* padding for final block */
	},

	/* CAAM_EE_ENCRYPT_BLOCK_CFB_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_CFB_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		true /* padding for final block */
	},

	/* CAAM_EE_ENCRYPT_BLOCK_OFB_IDX */
	[CAAM_EE_ENCRYPT_BLOCK_OFB_IDX] = {
		/* IV min,max,mod */
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MIN_MAX,
		CAAM_EE_DES_3DES_CBC_OFB_CFB_IV_MOD,
		/* tag min,max,mod */
		0, 0, 0,
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		false /* no padding for final block */
	}
};

static const struct caam_ee_cipher_desc caam_ee_cipher_descs[] = {
	/* CAAM_EE_CIPHER_AES(_IDX) */
	[CAAM_EE_CIPHER_AES_IDX] = {
			CAAM_EE_CIPHER_AES_KEY_LENGTH,
			CAAM_EE_AES_BLOCK_SIZE,
			CAAM_EE_WRAP_BLOCK_CCM_BMSK,
			caam_ee_wrap_blk_aes_descs,
			CAAM_EE_ENCRYPT_BLOCK_BMSKS_AES,
			caam_ee_encrypt_blk_aes_descs
	},

	/* CAAM_EE_CIPHER_DES(_IDX) */
	[CAAM_EE_CIPHER_DES_IDX] = {
			CAAM_EE_CIPHER_DES_KEY_LENGTH,
			CAAM_EE_DES_3DES_BLOCK_SIZE,
			0,
			NULL,
			CAAM_EE_ENCRYPT_BLOCK_BMSKS_DES_3DES,
			caam_ee_encrypt_blk_des_3des_descs
	},

	/* CAAM_EE_CIPHER_3DES(_IDX) */
	[CAAM_EE_CIPHER_3DES_IDX] = {
			CAAM_EE_CIPHER_3DES_KEY_LENGTH,
			CAAM_EE_DES_3DES_BLOCK_SIZE,
			0,
			NULL,
			CAAM_EE_ENCRYPT_BLOCK_BMSKS_DES_3DES,
			caam_ee_encrypt_blk_des_3des_descs
	}
};

/* sign algorithm (e.g. HMAC) descriptions */
struct caam_ee_sign_alg_desc {
	/* bitmask encoding the supported key lengths */
	uint32_t key_len_msk;
	/* supported hash modes */
	uint32_t hash_mode_msk;
};

/* hash mode descriptions */
struct caam_ee_hash_desc {
	/* tag or dgst length */
	uint32_t len_min;
	uint32_t len_max;
	/* the complete message must not be longer than this value */
	uint32_t max_total_len;
	/* the maximum amount of data one IOCTL can process at once */
	uint32_t max_block_len;
	/* block size */
	uint32_t block_size;
	/*
	 * All intermediate data blocks will be a multiple of the block size.
	 * In case the below value is true the last data block needs to
	 * be aligned too
	 */
	bool final_block_aligned;
};

#define CAAM_EE_SIGN_ALG_HMAC_KEY_LENGTH (CAAM_KEY_LEN_64BIT_BMSK |\
		CAAM_KEY_LEN_128BIT_BMSK |\
		CAAM_KEY_LEN_192BIT_BMSK |\
		CAAM_KEY_LEN_256BIT_BMSK)

#define CAAM_EE_SIGN_ALG_RSA_PKCS1V15_KEY_LENGTH (CAAM_KEY_LEN_1024BIT_BMSK |\
						  CAAM_KEY_LEN_2048BIT_BMSK |\
						  CAAM_KEY_LEN_4096BIT_BMSK)

#define CAAM_EE_HASH_HMAC_BMSKS (CAAM_EE_HASH_MD5_BMSK |\
		CAAM_EE_HASH_SHA1_BMSK |\
		CAAM_EE_HASH_SHA224_BMSK |\
		CAAM_EE_HASH_SHA256_BMSK)

#define CAAM_EE_HASH_RSA_PKCS1V15_BMSKS (CAAM_EE_HASH_MD5_BMSK |\
		CAAM_EE_HASH_SHA1_BMSK |\
		CAAM_EE_HASH_SHA224_BMSK |\
		CAAM_EE_HASH_SHA256_BMSK)

static const struct caam_ee_sign_alg_desc caam_ee_sign_alg_descs[] = {
	/* CAAM_EE_SIGN_ALG_HMAC_IDX	*/
	{
			CAAM_EE_SIGN_ALG_HMAC_KEY_LENGTH,
			CAAM_EE_HASH_HMAC_BMSKS
	},
	/* CAAM_EE_SIGN_ALG_RSA_PKCS1V15_IDX	*/
	{
			CAAM_EE_SIGN_ALG_RSA_PKCS1V15_KEY_LENGTH,
			CAAM_EE_HASH_RSA_PKCS1V15_BMSKS
	}
};

static const struct caam_ee_hash_desc caam_ee_hash_modes_descs[] = {
	/* CAAM_EE_HASH_MD5_IDX */
	[CAAM_EE_HASH_MD5_IDX] = {
		CAAM_EE_HASH_MD5_LEN_MIN,
		CAAM_EE_HASH_MD5_LEN_MAX, /* dgst/tag length */
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		CAAM_EE_HASH_ALIGN,  /* align data blocks to 64 */
		false /* no final padding required */
	},
	/* CAAM_EE_HASH_SHA1_IDX */
	[CAAM_EE_HASH_SHA1_IDX] = {
		CAAM_EE_HASH_SHA1_LEN_MIN,
		CAAM_EE_HASH_SHA1_LEN_MAX, /* dgst/tag length */
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		CAAM_EE_HASH_ALIGN,  /* align data blocks to 64 */
		false /* no final padding required */
	},
	/* CAAM_EE_HASH_SHA224_IDX */
	[CAAM_EE_HASH_SHA224_IDX] =	{
		CAAM_EE_HASH_SHA224_LEN_MIN,
		CAAM_EE_HASH_SHA224_LEN_MAX, /* dgst/tag length */
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		CAAM_EE_HASH_ALIGN,  /* align data blocks to 64 */
		false /* no final padding required */
	},
	/* CAAM_EE_HASH_SHA256_IDX */
	[CAAM_EE_HASH_SHA256_IDX] = {
		CAAM_EE_HASH_SHA256_LEN_MIN,
		CAAM_EE_HASH_SHA256_LEN_MAX, /* dgst/tag length */
		CAAM_EE_MAX_TOTAL_LEN, /* max_total_len */
		CAAM_EE_MAX_DATA_LEN,  /* max_block_len */
		CAAM_EE_HASH_ALIGN,  /* align data blocks to 64 */
		false /* no final padding required */
	}
};

#endif /* _CAAM_EE_H */
